"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.RecommendationServiceImpl = exports.IgnoreBehavior = exports.COMMAND_MARKDOWN_API_RENDER = exports.filterUnique = void 0;
/*-----------------------------------------------------------------------------------------------
 *  Copyright (c) Red Hat, Inc. All rights reserved.
 *  Licensed under the EPL v2.0 License. See LICENSE file in the project root for license information.
 *-----------------------------------------------------------------------------------------------*/
const path_1 = __importDefault(require("path"));
const vscode_1 = require("vscode");
const recommendationModel_1 = require("../recommendationModel");
const storageServiceImpl_1 = require("./storageServiceImpl");
const vscodeUtil_1 = require("./vscodeUtil");
const filterUnique = (value, index, self) => self.indexOf(value) === index;
exports.filterUnique = filterUnique;
/**
* Command string to render markdown string to html string
*/
exports.COMMAND_MARKDOWN_API_RENDER = 'markdown.api.render';
var IgnoreBehavior;
(function (IgnoreBehavior) {
    IgnoreBehavior["StartupOnly"] = "StartupOnly";
    IgnoreBehavior["All"] = "All";
    IgnoreBehavior["SingleRecommendation"] = "SingleRecommendation";
})(IgnoreBehavior = exports.IgnoreBehavior || (exports.IgnoreBehavior = {}));
class RecommendationServiceImpl {
    constructor(context, telemetryService) {
        this.extensionContext = context;
        this.telemetryService = telemetryService;
        this.storageService = this.createStorageService(context);
    }
    createStorageService(context) {
        const storagePath = this.getRecommendationWorkingDir(context);
        return new storageServiceImpl_1.StorageServiceImpl(storagePath);
    }
    getRecommendationWorkingDir(context) {
        return path_1.default.resolve(context.globalStorageUri.fsPath, '..', 'vscode-extension-recommender');
    }
    register(toAdd) {
        return __awaiter(this, void 0, void 0, function* () {
            const newSession = yield this.addRecommendationsToModel(toAdd);
            if (newSession) {
                // Return fast (ie, don't await) so as not to slow down caller
                this.showStartupRecommendations();
            }
        });
    }
    ignoreRecommendations() {
        const ignore = vscode_1.workspace.getConfiguration().get('extensions.ignoreRecommendations');
        if (ignore) {
            return true;
        }
        return false;
    }
    create(extensionId, extensionDisplayName, description, shouldShowOnStartup) {
        return {
            sourceId: this.extensionContext.extension.id,
            extensionId: extensionId,
            extensionDisplayName: extensionDisplayName,
            description: description,
            shouldShowOnStartup: shouldShowOnStartup,
            timestamp: Date.now(),
            userIgnored: false
        };
    }
    addRecommendationsToModel(toAdd) {
        return __awaiter(this, void 0, void 0, function* () {
            const newSession = yield this.storageService.runWithLock((model) => __awaiter(this, void 0, void 0, function* () {
                const current = model.recommendations;
                const newRecs = [];
                const toAddAlreadyAdded = [];
                for (let i = 0; i < current.length; i++) {
                    const beingAdded = this.findRecommendation(current[i].sourceId, current[i].extensionId, toAdd);
                    if (beingAdded) {
                        beingAdded.userIgnored = current[i].userIgnored;
                        newRecs.push(beingAdded);
                        toAddAlreadyAdded.push(beingAdded);
                    }
                    else {
                        newRecs.push(current[i]);
                    }
                }
                for (let i = 0; i < toAdd.length; i++) {
                    if (!toAddAlreadyAdded.includes(toAdd[i])) {
                        newRecs.push(toAdd[i]);
                    }
                }
                model.recommendations = newRecs;
                return model;
            }));
            return newSession;
        });
    }
    findRecommendation(needleFrom, needleTo, haystack) {
        for (let i = 0; i < haystack.length; i++) {
            if (haystack[i].sourceId === needleFrom &&
                haystack[i].extensionId === needleTo) {
                return haystack[i];
            }
        }
        return undefined;
    }
    show(toExtension, ignoreTimelock, overrideDescription, level, hideNever) {
        return __awaiter(this, void 0, void 0, function* () {
            // Show a single recommendation immediately, if certain conditions are met
            // Specifically, if the recommender is installed, and the recommended is not installed, 
            // and the recommended has not been timelocked in this session or ignored by user previously
            if (this.ignoreRecommendations()) {
                return;
            }
            const fromExtension = this.extensionContext.extension.id;
            if ((0, vscodeUtil_1.isExtensionInstalled)(fromExtension) && !(0, vscodeUtil_1.isExtensionInstalled)(toExtension)) {
                const model = yield this.storageService.readRecommendationModel();
                if (model && (ignoreTimelock || !model.timelocked.includes(toExtension))) {
                    const rec = model.recommendations.find((x) => x.extensionId === toExtension && x.sourceId === fromExtension);
                    if (rec && !rec.userIgnored) {
                        const displayName = rec.extensionDisplayName || rec.extensionId;
                        const recToUse = Object.assign({}, rec);
                        if (overrideDescription) {
                            recToUse.description = overrideDescription;
                        }
                        const recommendationsForId = model.recommendations.filter((x) => x.extensionId === toExtension)
                            .filter((x) => (0, vscodeUtil_1.isExtensionInstalled)(x.sourceId));
                        const hideNeverVal = (hideNever === undefined ? false : hideNever);
                        const msg = this.collectShowNowMessage(toExtension, displayName, recToUse, recommendationsForId);
                        this.displaySingleRecommendation(toExtension, displayName, [recToUse.sourceId], msg, level || recommendationModel_1.Level.Info, IgnoreBehavior.SingleRecommendation, hideNeverVal, rec);
                    }
                }
            }
            return undefined;
        });
    }
    showStartupRecommendations() {
        return __awaiter(this, void 0, void 0, function* () {
            // wait 6 seconds for other tools to start up
            yield new Promise(resolve => setTimeout(resolve, 6000));
            // Then show the dialogs
            if (this.ignoreRecommendations()) {
                return;
            }
            const model = yield this.storageService.readRecommendationModel();
            if (model) {
                const recommendedExtension = model.recommendations
                    .map((x) => x.extensionId)
                    .filter(exports.filterUnique)
                    .filter((x) => !(0, vscodeUtil_1.isExtensionInstalled)(x));
                for (let i = 0; i < recommendedExtension.length; i++) {
                    this.showStartupRecommendationsForSingleExtension(model, recommendedExtension[i]);
                }
            }
        });
    }
    showStartupRecommendationsForSingleExtension(model, id) {
        return __awaiter(this, void 0, void 0, function* () {
            const startupRecommendationsForId = model.recommendations.filter((x) => x.extensionId === id)
                .filter((x) => (0, vscodeUtil_1.isExtensionInstalled)(x.sourceId))
                .filter((x) => x.shouldShowOnStartup);
            const ignoredCount = startupRecommendationsForId.filter((x) => x.userIgnored === true).length;
            const allIgnored = startupRecommendationsForId.length === ignoredCount;
            const count = startupRecommendationsForId.length;
            if (count === 0 || allIgnored)
                return;
            const displayName = this.findMode(startupRecommendationsForId.map((x) => x.extensionDisplayName)) || id;
            const msg = this.collectMessage(id, displayName, startupRecommendationsForId);
            const sourceIds = startupRecommendationsForId.map((x) => x.sourceId);
            this.displaySingleRecommendation(id, displayName, sourceIds, msg, recommendationModel_1.Level.Info, IgnoreBehavior.StartupOnly, false);
        });
    }
    safeDescriptionWithPeriod(description) {
        const trimmed = description.trim();
        const lastChar = trimmed.charAt(trimmed.length - 1);
        if (![".", "!", "?"].includes(lastChar)) {
            return trimmed + ".";
        }
        return description;
    }
    linkToMore(id) {
        const obj = {
            id: id
        };
        const encoded = encodeURIComponent(JSON.stringify(obj));
        const tellMeMore1 = `[Learn More...](command:${this.getSingleMarkdownCommandId()}?${encoded})`;
        return tellMeMore1;
    }
    collectMessage(id, displayName, recommendationsForId) {
        const count = recommendationsForId.length;
        if (count === 1) {
            const fromExtensionId = recommendationsForId[0].sourceId;
            const fromExtensionName = (0, vscodeUtil_1.getInstalledExtensionName)(fromExtensionId) || fromExtensionId;
            const safeDesc = this.safeDescriptionWithPeriod(recommendationsForId[0].description);
            const msg = `${fromExtensionName} recommends you install "${displayName}":\n${safeDesc} `;
            return msg;
        }
        else if (count > 1) {
            return this.collectMultiCountMessage(id, displayName, recommendationsForId);
        }
        else {
            return "An unknown extension recommends that you also install \"" + displayName + "\".";
        }
    }
    collectShowNowMessage(id, displayName, primary, recommendationsForId) {
        const fromExtensionId = primary.sourceId;
        const fromExtensionName = (0, vscodeUtil_1.getInstalledExtensionName)(fromExtensionId) || fromExtensionId;
        const msg = `"${fromExtensionName}" recommends you install "${displayName}":\n"${recommendationsForId[0].description}" `;
        const count = recommendationsForId.length;
        if (count === 1) {
            return msg;
        }
        else if (count > 1) {
            const clone = [...recommendationsForId].filter((x) => x.sourceId !== primary.sourceId || x.extensionId !== primary.extensionId);
            const singlePlural = clone.length > 1 ? "extensions also recommend" : "extension also recommends";
            const others = `(${clone.length} other ${singlePlural} this. ${this.linkToMore(id)})`;
            return msg + others;
        }
        else {
            return "An unknown extension recommends that you also install \"" + displayName + "\".";
        }
    }
    collectMultiCountMessage(id, displayName, recommendationsForId) {
        const ignoredList = recommendationsForId.filter((x) => x.userIgnored);
        const notIgnoredList = recommendationsForId.filter((x) => x.userIgnored === false);
        const ignoredCount = ignoredList.length;
        const notIgnoredCount = notIgnoredList.length;
        let countMessage = "";
        if (ignoredCount === 0 && notIgnoredCount > 0) {
            const singlePlural = notIgnoredCount > 1 ? "extensions recommend" : "extension recommends";
            countMessage = `${notIgnoredCount} ${singlePlural} you install "${displayName}". `;
        }
        else if (notIgnoredCount > 0 && ignoredCount > 0) {
            // one or more of ignored and not-ignored / new recommendations
            const rec = recommendationsForId.length > 1 ? "recommend" : "recommends";
            const spTotal = (notIgnoredCount + ignoredCount) > 1 ? "extensions" : "extension";
            countMessage = `${recommendationsForId.length} ${spTotal} (${notIgnoredCount} new) ${rec} you install "${displayName}". `;
        }
        const recommenderNames = recommendationsForId.map((x) => {
            const fromExtensionId = x.sourceId;
            const fromExtensionName = (0, vscodeUtil_1.getInstalledExtensionName)(fromExtensionId) || fromExtensionId;
            return fromExtensionName;
        });
        const lastName = "\"" + recommenderNames[recommenderNames.length - 1] + "\"";
        const withoutLast = recommenderNames.slice(0, -1).map((x) => "\"" + x + "\"");
        const withoutLastAddCommas = withoutLast.join(", ");
        const finalMsg = countMessage + "The recommending extensions are " + withoutLastAddCommas + " and " + lastName + ". ";
        return finalMsg + this.linkToMore(id);
    }
    findMode(arr) {
        return arr.sort((a, b) => arr.filter(v => v === a).length - arr.filter(v => v === b).length).pop();
    }
    displaySingleRecommendation(id, extensionDisplayName, recommenderList, msg, level, ignoreBehavior, hideNever, singleRec) {
        return __awaiter(this, void 0, void 0, function* () {
            // Ensure command is registered before prompting the user
            this.registerSingleMarkdownCommand();
            // Timelock this regardless of what the user selects.
            yield this.timelockRecommendationFor(id);
            const choice = yield (0, vscodeUtil_1.promptUserUtil)(msg, level, hideNever);
            const safeChoice = choice ? choice : 'canceled';
            this.fireTelemetrySuccess(id, recommenderList, safeChoice);
            if (choice) {
                if (choice === recommendationModel_1.UserChoice.Never) {
                    if (ignoreBehavior === IgnoreBehavior.All) {
                        yield this.markIgnored(id, false);
                    }
                    else if (ignoreBehavior === IgnoreBehavior.StartupOnly) {
                        yield this.markIgnored(id, true);
                    }
                    else if (ignoreBehavior === IgnoreBehavior.SingleRecommendation && singleRec) {
                        yield this.markIgnoredSingleRec(singleRec);
                    }
                }
                else {
                    if (choice === recommendationModel_1.UserChoice.Install) {
                        yield (0, vscodeUtil_1.installExtensionUtil)(id, extensionDisplayName, 30000);
                    }
                }
            }
            return choice;
        });
    }
    timelockRecommendationFor(id) {
        return __awaiter(this, void 0, void 0, function* () {
            yield this.storageService.runWithLock((model) => __awaiter(this, void 0, void 0, function* () {
                if (!model.timelocked.includes(id)) {
                    model.timelocked.push(id);
                    return model;
                }
                return undefined;
            }));
        });
    }
    registerSingleMarkdownCommand() {
        return __awaiter(this, void 0, void 0, function* () {
            const commandId = this.getSingleMarkdownCommandId();
            const ret = yield vscode_1.commands.getCommands();
            if (!ret.includes(commandId)) {
                try {
                    const cmdResult = vscode_1.commands.registerCommand(commandId, (param) => __awaiter(this, void 0, void 0, function* () {
                        if (param && param.id) {
                            this.runShowMarkdownCommand(param.id);
                        }
                        else {
                            vscode_1.window.showInformationMessage("Unable to show recommendation report for extension 'undefined'.");
                        }
                    }));
                    this.extensionContext.subscriptions.push(cmdResult);
                }
                catch (err) {
                    // Do nothing, might be a race condition / duplicate
                }
            }
        });
    }
    getSingleMarkdownCommandId() {
        return "_vscode-extension-recommender.showMarkdown";
    }
    runShowMarkdownCommand(id) {
        return __awaiter(this, void 0, void 0, function* () {
            const model = yield this.storageService.readRecommendationModel();
            if (model) {
                const recommendedExtension = model.recommendations
                    .filter((x) => x.extensionId === id)
                    .filter(exports.filterUnique)
                    .filter((x) => !(0, vscodeUtil_1.isExtensionInstalled)(x.extensionId));
                const displayName = this.findMode(recommendedExtension.map((x) => x.extensionDisplayName)) || id;
                const sorted = recommendedExtension.sort((x, y) => {
                    if (x.shouldShowOnStartup === y.shouldShowOnStartup) {
                        return x.userIgnored === y.userIgnored ? 0 : x.userIgnored ? 1 : -1;
                    }
                    return x.shouldShowOnStartup ? -1 : 1;
                });
                const header = `# Extensions recommending "${displayName}"\n`;
                const lines = [];
                for (let i = 0; i < sorted.length; i++) {
                    const r = sorted[i];
                    lines.push("## " + (0, vscodeUtil_1.getInstalledExtensionName)(r.sourceId));
                    lines.push("Reason: " + r.description);
                    if (r.userIgnored) {
                        lines.push("- This recommendation was previously ignored by the user.");
                    }
                    if (!r.shouldShowOnStartup) {
                        lines.push("- This recommendation is triggered only in specific situations or workflows.");
                    }
                }
                const mdString = header + lines.join("\n");
                const path = yield this.storageService.writeKey(id, mdString);
                if (path)
                    vscode_1.commands.executeCommand("markdown.showPreview", vscode_1.Uri.parse(path));
                else {
                    // idk, show warning?
                }
            }
        });
    }
    markIgnored(id, startupOnly) {
        return __awaiter(this, void 0, void 0, function* () {
            // Mark all CURRENT (not future, from a new unknown extension) 
            // recommendations to the given id. 
            const newSession = yield this.storageService.runWithLock((model) => __awaiter(this, void 0, void 0, function* () {
                const current = model.recommendations;
                for (let i = 0; i < current.length; i++) {
                    if (current[i].extensionId === id) {
                        if (!startupOnly || current[i].shouldShowOnStartup)
                            current[i].userIgnored = true;
                    }
                }
                return model;
            }));
        });
    }
    markIgnoredSingleRec(rec) {
        return __awaiter(this, void 0, void 0, function* () {
            // Mark all CURRENT (not future, from a new unknown extension) 
            // recommendations to the given id. 
            const newSession = yield this.storageService.runWithLock((model) => __awaiter(this, void 0, void 0, function* () {
                const current = model.recommendations;
                for (let i = 0; i < current.length; i++) {
                    if (current[i].extensionId === rec.extensionId && current[i].sourceId === rec.sourceId) {
                        current[i].userIgnored = true;
                    }
                }
                return model;
            }));
        });
    }
    fireTelemetrySuccess(target, recommenderList, choice) {
        return __awaiter(this, void 0, void 0, function* () {
            if (this.telemetryService) {
                this.telemetryService.send({
                    name: "recommendation",
                    properties: {
                        recommendation: target,
                        recommenders: recommenderList,
                        choice: choice.toString()
                    }
                });
            }
        });
    }
}
exports.RecommendationServiceImpl = RecommendationServiceImpl;
//# sourceMappingURL=recommendationServiceImpl.js.map