"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : new P(function (resolve) { resolve(result.value); }).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
Object.defineProperty(exports, "__esModule", { value: true });
const messages_1 = require("../protocol/generated/messages");
const common_1 = require("./common");
const outgoing_1 = require("../protocol/generated/outgoing");
/**
 * Server creation utility
 */
class ServerCreation {
    /**
     * Constructs a new server model handler
     * @param connection message connection to the RSP
     * @param emitter event emitter to handle notification events
     */
    constructor(connection, emitter) {
        this.connection = connection;
        this.emitter = emitter;
    }
    /**
     * Sends a request to create a server from a given directory, subscribe to the 'serverAdded'
     * event to see when the server creation is finished
     * @param path path to the server's root directory
     * @param id unique identifier for the newly created server
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromPathAsync(path, id, attributes, timeout = common_1.Common.DEFAULT_TIMEOUT) {
        return __awaiter(this, void 0, void 0, function* () {
            const serverBeans = yield common_1.Common.sendSimpleRequest(this.connection, messages_1.Messages.Server.FindServerBeansRequest.type, { filepath: path }, timeout / 2, outgoing_1.ErrorMessages.FINDSERVERBEANS_TIMEOUT);
            const atts = Object.assign({}, attributes);
            atts['server.home.dir'] = serverBeans[0].location;
            if ((serverBeans[0].typeCategory === 'MINISHIFT') || (serverBeans[0].typeCategory === 'CDK')) {
                atts['server.home.file'] = serverBeans[0].location;
            }
            const serverAttributes = {
                id: id,
                serverType: serverBeans[0].serverAdapterTypeId,
                attributes: atts
            };
            return common_1.Common.sendSimpleRequest(this.connection, messages_1.Messages.Server.CreateServerRequest.type, serverAttributes, timeout, outgoing_1.ErrorMessages.CREATESERVER_TIMEOUT);
        });
    }
    /**
     * Sends a request to create a server from an existing ServerBean object, subscribe to the 'serverAdded'
     * event to see when the server creation is finished
     * @param serverBean ServerBean object
     * @param id unique identifier for the new server, if left empty, the serverBean.name will be used
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromBeanAsync(serverBean, id, attributes, timeout = common_1.Common.DEFAULT_TIMEOUT) {
        return __awaiter(this, void 0, void 0, function* () {
            const serverId = id ? id : serverBean.name;
            const atts = Object.assign({}, attributes);
            atts['server.home.dir'] = serverBean.location;
            if ((serverBean.typeCategory === 'MINISHIFT') || (serverBean.typeCategory === 'CDK')) {
                atts['server.home.file'] = serverBean.location;
            }
            const serverAttributes = {
                id: serverId,
                serverType: serverBean.serverAdapterTypeId,
                attributes: atts
            };
            return common_1.Common.sendSimpleRequest(this.connection, messages_1.Messages.Server.CreateServerRequest.type, serverAttributes, timeout, outgoing_1.ErrorMessages.CREATESERVER_TIMEOUT);
        });
    }
    /**
     * Sends a request to create a server from a given directory, then waits for the 'serverAdded'
     * event with the given id
     * @param path path to the server's root directory
     * @param id unique identifier for the newly created server
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromPath(path, id, attributes, timeout = common_1.Common.DEFAULT_TIMEOUT) {
        return new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
            const timer = setTimeout(() => {
                return reject(new Error(outgoing_1.ErrorMessages.CREATESERVER_TIMEOUT));
            }, timeout);
            let result;
            const listener = (handle) => {
                if (handle.id === id) {
                    result.then(status => {
                        clearTimeout(timer);
                        this.emitter.removeListener('serverAdded', listener);
                        resolve(handle);
                    });
                }
            };
            this.emitter.prependListener('serverAdded', listener);
            const serverBeans = yield this.connection.sendRequest(messages_1.Messages.Server.FindServerBeansRequest.type, { filepath: path });
            const atts = Object.assign({}, attributes);
            atts['server.home.dir'] = serverBeans[0].location;
            if ((serverBeans[0].typeCategory === 'MINISHIFT') || (serverBeans[0].typeCategory === 'CDK')) {
                atts['server.home.file'] = serverBeans[0].location;
            }
            const serverAttributes = {
                id: id,
                serverType: serverBeans[0].serverAdapterTypeId,
                attributes: atts
            };
            result = this.connection.sendRequest(messages_1.Messages.Server.CreateServerRequest.type, serverAttributes);
        }));
    }
    /**
     * Sends a request to create a server from an existing ServerBean object,  then waits for the 'serverAdded'
     * event with the given id
     * @param serverBean ServerBean object
     * @param id unique identifier for the new server, if left empty, the serverBean.name will be used
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromBean(serverBean, id, attributes, timeout = common_1.Common.DEFAULT_TIMEOUT) {
        return new Promise((resolve, reject) => __awaiter(this, void 0, void 0, function* () {
            const serverId = id ? id : serverBean.name;
            const timer = setTimeout(() => {
                return reject(new Error(outgoing_1.ErrorMessages.CREATESERVER_TIMEOUT));
            }, timeout);
            let result;
            const listener = (handle) => {
                if (handle.id === serverId) {
                    result.then(status => {
                        clearTimeout(timer);
                        this.emitter.removeListener('serverAdded', listener);
                        resolve(handle);
                    });
                }
            };
            this.emitter.prependListener('serverAdded', listener);
            const atts = Object.assign({}, attributes);
            atts['server.home.dir'] = serverBean.location;
            if ((serverBean.typeCategory === 'MINISHIFT') || (serverBean.typeCategory === 'CDK')) {
                atts['server.home.file'] = serverBean.location;
            }
            const serverAttributes = {
                id: serverId,
                serverType: serverBean.serverAdapterTypeId,
                attributes: atts
            };
            if ((serverBean.typeCategory === 'MINISHIFT') || (serverBean.typeCategory === 'CDK')) {
                serverAttributes.attributes['server.home.file'] = serverBean.location;
            }
            result = this.connection.sendRequest(messages_1.Messages.Server.CreateServerRequest.type, serverAttributes);
        }));
    }
}
exports.ServerCreation = ServerCreation;
//# sourceMappingURL=serverCreation.js.map