/// <reference types="node" />
import { MessageConnection } from 'vscode-jsonrpc';
import { Protocol } from '../protocol/generated/protocol';
import { EventEmitter } from 'events';
/**
 * Server creation utility
 */
export declare class ServerCreation {
    private connection;
    private emitter;
    /**
     * Constructs a new server model handler
     * @param connection message connection to the RSP
     * @param emitter event emitter to handle notification events
     */
    constructor(connection: MessageConnection, emitter: EventEmitter);
    /**
     * Sends a request to create a server from a given directory, subscribe to the 'serverAdded'
     * event to see when the server creation is finished
     * @param path path to the server's root directory
     * @param id unique identifier for the newly created server
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromPathAsync(path: string, id: string, attributes?: {
        [index: string]: any;
    }, timeout?: number): Promise<Protocol.CreateServerResponse>;
    /**
     * Sends a request to create a server from an existing ServerBean object, subscribe to the 'serverAdded'
     * event to see when the server creation is finished
     * @param serverBean ServerBean object
     * @param id unique identifier for the new server, if left empty, the serverBean.name will be used
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromBeanAsync(serverBean: Protocol.ServerBean, id?: string, attributes?: {
        [index: string]: any;
    }, timeout?: number): Promise<Protocol.CreateServerResponse>;
    /**
     * Sends a request to create a server from a given directory, then waits for the 'serverAdded'
     * event with the given id
     * @param path path to the server's root directory
     * @param id unique identifier for the newly created server
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromPath(path: string, id: string, attributes?: {
        [index: string]: any;
    }, timeout?: number): Promise<Protocol.ServerHandle>;
    /**
     * Sends a request to create a server from an existing ServerBean object,  then waits for the 'serverAdded'
     * event with the given id
     * @param serverBean ServerBean object
     * @param id unique identifier for the new server, if left empty, the serverBean.name will be used
     * @param attributes optional extra server attributes
     * @param timeout timeout in milliseconds
     */
    createServerFromBean(serverBean: Protocol.ServerBean, id?: string, attributes?: {
        [index: string]: any;
    }, timeout?: number): Promise<Protocol.ServerHandle>;
}
