"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Helper class for sending requests and notifications
 */
class Common {
    /**
     * Template method for sending requests and receiving the response
     * @param connection the message connection to RSP server
     * @param messageType type of the message being sent
     * @param payload parameters of the message being sent
     * @param timeout timeout in milliseconds
     * @param timeoutMessage error message in case of timeout
     */
    static sendSimpleRequest(connection, messageType, payload, timeout, timeoutMessage) {
        return new Promise((resolve, reject) => {
            const timer = setTimeout(() => {
                reject(new Error(timeoutMessage));
            }, timeout);
            return connection.sendRequest(messageType, payload).then(result => {
                clearTimeout(timer);
                resolve(result);
            });
        });
    }
    /**

     * Template method for sending notifications and synchronously waiting for a response by subscribing to an event
     * @param connection the message connection to RSP server
     * @param messageType type of the message being sent
     * @param payload parameters of the message being sent
     * @param emitter event emitter used to subscribe for the response event
     * @param eventId id/name of the response event to wait for
     * @param listener callback to handle the response event parameters
     * @param timeout timeout in milliseconds
     * @param timeoutMessage error message in case of timeout
     */
    static sendRequestSync(connection, messageType, payload, emitter, eventId, listener, timeout, timeoutMessage) {
        return new Promise((resolve, reject) => {
            const timer = setTimeout(() => {
                return reject(new Error(timeoutMessage));
            }, timeout);
            let response;
            const handler = (params) => {
                if (listener(params)) {
                    response.then(() => {
                        clearTimeout(timer);
                        emitter.removeListener(eventId, listener);
                        resolve(params);
                    });
                }
            };
            emitter.prependListener(eventId, handler);
            response = connection.sendRequest(messageType, payload);
        });
    }
    /**
     * Template method for sending simple notifications to the server
     * @param connection message connection to the server
     * @param messageType type of the notification being sent
     * @param payload payload (parameters) of the message being sent
     */
    static sendSimpleNotification(connection, messageType, payload) {
        connection.sendNotification(messageType, payload);
    }
}
Common.SHORT_TIMEOUT = 2000;
Common.DEFAULT_TIMEOUT = 7500;
Common.LONG_TIMEOUT = 20000;
Common.VERY_LONG_TIMEOUT = 60000;
exports.Common = Common;
//# sourceMappingURL=common.js.map