"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const net = require("net");
const rpc = require("vscode-jsonrpc");
const messages_1 = require("./protocol/generated/messages");
const serverCreation_1 = require("./util/serverCreation");
const events_1 = require("events");
const incoming_1 = require("./protocol/generated/incoming");
const outgoing_1 = require("./protocol/generated/outgoing");
const outgoingsync_1 = require("./util/outgoingsync");
const common_1 = require("./util/common");
/**
 * Runtime Server Protocol client implementation using JSON RPC
 */
class RSPClient {
    /**
     * Constructs a new RSP client
     * @param host hostname/address to connect to
     * @param port port of the running RSP service
     */
    constructor(host, port) {
        this.host = host;
        this.port = port;
        this.emitter = new events_1.EventEmitter();
    }
    /**
     * Initiates connection to the RSP server
     *
     * @param timeout operation timeout in milliseconds, default 2000 ms
     */
    connect(timeout = common_1.Common.DEFAULT_TIMEOUT) {
        return new Promise((resolve, reject) => {
            const timer = setTimeout(() => {
                return reject(new Error(`Failed to establish connection to ${this.host}:${this.port} within time`));
            }, timeout);
            this.socket = net.connect(this.port, this.host);
            this.socket.on('close', () => {
                this.emitter.emit('connectionClosed');
            });
            this.socket.on('connect', () => {
                this.connection = rpc.createMessageConnection(new rpc.StreamMessageReader(this.socket), new rpc.StreamMessageWriter(this.socket));
                if (this.connection.trace) {
                    this.connection.trace(rpc.Trace.Verbose, { log: (message, data) => {
                            console.log(`Message=${message} data=${data}`);
                        } });
                }
                this.connection.listen();
                this.serverUtil = new serverCreation_1.ServerCreation(this.connection, this.emitter);
                this.incoming = new incoming_1.Incoming(this.connection, this.emitter);
                this.outgoing = new outgoing_1.Outgoing(this.connection);
                this.outgoingSync = new outgoingsync_1.OutgoingSynchronous(this.connection, this.emitter);
                clearTimeout(timer);
                resolve();
            });
        });
    }
    onConnectionClosed(listener) {
        this.emitter.on('connectionClosed', listener);
    }
    removeOnConnectionClosed(listener) {
        this.emitter.removeListener('connectionClosed', listener);
    }
    /**
     * Terminates an existing connection
     *
     * @throws {@link rpc.ConnectionError} if connection is not initialized or already disposed
     */
    disconnect() {
        if (!this.connection) {
            throw new rpc.ConnectionError(rpc.ConnectionErrors.Closed, 'Connection not initialized');
        }
        this.emitter.removeAllListeners();
        this.connection.dispose();
        this.socket.end();
        this.socket.destroy();
    }
    /**
     * Terminates the currently running RSP server instance and disconnects itself
     */
    shutdownServer() {
        this.connection.sendNotification(messages_1.Messages.Server.ShutdownNotification.type);
        this.disconnect();
    }
    getIncomingHandler() {
        return this.incoming;
    }
    getOutgoingHandler() {
        return this.outgoing;
    }
    getOutgoingSyncHandler() {
        return this.outgoingSync;
    }
    getServerCreation() {
        return this.serverUtil;
    }
    /**
     * Returns the capabilities implemented by the client
     */
    getCapabilities() {
        return { map: { 'protocol.version': '0.21.0', 'prompt.string': 'true' } };
    }
    /**
     * Retrieves all listeners bound to an event
     *
     * @param eventName name of the event to get listeners for
     */
    getListeners(eventName) {
        return this.emitter.listeners(eventName);
    }
}
exports.RSPClient = RSPClient;
//# sourceMappingURL=client.js.map