"use strict";
var CorrelationContextManager = require("./AutoCollection/CorrelationContextManager"); // Keep this first
var AutoCollectConsole = require("./AutoCollection/Console");
var AutoCollectExceptions = require("./AutoCollection/Exceptions");
var AutoCollectPerformance = require("./AutoCollection/Performance");
var AutoCollectClientRequests = require("./AutoCollection/ClientRequests");
var AutoCollectServerRequests = require("./AutoCollection/ServerRequests");
var Client = require("./Library/Client");
var Logging = require("./Library/Logging");
/**
 * The singleton meta class for the default client of the client. This class is used to setup/start and configure
 * the auto-collection behavior of the application insights module.
 */
var ApplicationInsights = (function () {
    function ApplicationInsights() {
    }
    /**
     * Initializes a client with the given instrumentation key, if this is not specified, the value will be
     * read from the environment variable APPINSIGHTS_INSTRUMENTATIONKEY
     * @returns {ApplicationInsights/Client} a new client
     */
    ApplicationInsights.getClient = function (instrumentationKey) {
        return new Client(instrumentationKey);
    };
    /**
     * Initializes the default client of the client and sets the default configuration
     * @param instrumentationKey the instrumentation key to use. Optional, if this is not specified, the value will be
     * read from the environment variable APPINSIGHTS_INSTRUMENTATIONKEY
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setup = function (instrumentationKey) {
        if (!ApplicationInsights.client) {
            ApplicationInsights.client = ApplicationInsights.getClient(instrumentationKey);
            ApplicationInsights._console = new AutoCollectConsole(ApplicationInsights.client);
            ApplicationInsights._exceptions = new AutoCollectExceptions(ApplicationInsights.client);
            ApplicationInsights._performance = new AutoCollectPerformance(ApplicationInsights.client);
            ApplicationInsights._serverRequests = new AutoCollectServerRequests(ApplicationInsights.client);
            ApplicationInsights._clientRequests = new AutoCollectClientRequests(ApplicationInsights.client);
        }
        else {
            Logging.info("The default client is already setup");
        }
        if (ApplicationInsights.client && ApplicationInsights.client.channel) {
            ApplicationInsights.client.channel.setOfflineMode(ApplicationInsights._isOfflineMode);
        }
        return ApplicationInsights;
    };
    /**
     * Starts automatic collection of telemetry. Prior to calling start no telemetry will be collected
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.start = function () {
        if (!!this.client) {
            ApplicationInsights._isStarted = true;
            ApplicationInsights._console.enable(ApplicationInsights._isConsole);
            ApplicationInsights._exceptions.enable(ApplicationInsights._isExceptions);
            ApplicationInsights._performance.enable(ApplicationInsights._isPerformance);
            ApplicationInsights._serverRequests.useAutoCorrelation(ApplicationInsights._isCorrelating);
            ApplicationInsights._serverRequests.enable(ApplicationInsights._isRequests);
            ApplicationInsights._clientRequests.enable(ApplicationInsights._isDependencies);
        }
        else {
            Logging.warn("Start cannot be called before setup");
        }
        return ApplicationInsights;
    };
    /**
     * Returns an object that is shared across all code handling a given request. This can be used similarly to thread-local storage in other languages.
     * Properties set on this object will be available to telemetry processors.
     *
     * Do not store sensitive information here.
     * Custom properties set on this object can be exposed in a future SDK release via outgoing HTTP headers.
     * This is to allow for correlating data cross-component.
     *
     * This method will return null if automatic dependency correlation is disabled.
     * @returns A plain object for request storage or null if automatic dependency correlation is disabled.
     */
    ApplicationInsights.getCorrelationContext = function () {
        if (this._isCorrelating) {
            return CorrelationContextManager.CorrelationContextManager.getCurrentContext();
        }
        return null;
    };
    /**
     * Returns a function that will get the same correlation context within its function body as the code executing this function.
     * Use this method if automatic dependency correlation is not propagating correctly to an asynchronous callback.
     */
    ApplicationInsights.wrapWithCorrelationContext = function (fn) {
        return CorrelationContextManager.CorrelationContextManager.wrapCallback(fn);
    };
    /**
     * Sets the state of console tracking (enabled by default)
     * @param value if true console activity will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectConsole = function (value) {
        ApplicationInsights._isConsole = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._console.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of exception tracking (enabled by default)
     * @param value if true uncaught exceptions will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectExceptions = function (value) {
        ApplicationInsights._isExceptions = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._exceptions.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of performance tracking (enabled by default)
     * @param value if true performance counters will be collected every second and sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectPerformance = function (value) {
        ApplicationInsights._isPerformance = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._performance.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of request tracking (enabled by default)
     * @param value if true requests will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectRequests = function (value) {
        ApplicationInsights._isRequests = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._serverRequests.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of dependency tracking (enabled by default)
     * @param value if true dependencies will be sent to Application Insights
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoCollectDependencies = function (value) {
        ApplicationInsights._isDependencies = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._clientRequests.enable(value);
        }
        return ApplicationInsights;
    };
    /**
     * Sets the state of automatic dependency correlation (enabled by default)
     * @param value if true dependencies will be correlated with requests
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.setAutoDependencyCorrelation = function (value) {
        ApplicationInsights._isCorrelating = value;
        if (ApplicationInsights._isStarted) {
            ApplicationInsights._serverRequests.useAutoCorrelation(value);
        }
        return ApplicationInsights;
    };
    /**
    * Enable or disable offline mode to cache events when client is offline (disabled by default)
    * @param value if true events that occured while client is offline will be cached on disk
    * @param resendInterval. The wait interval for resending cached events.
    * @returns {ApplicationInsights} this class
    */
    ApplicationInsights.setOfflineMode = function (value, resendInterval) {
        ApplicationInsights._isOfflineMode = value;
        if (ApplicationInsights.client && ApplicationInsights.client.channel) {
            ApplicationInsights.client.channel.setOfflineMode(value, resendInterval);
        }
        return ApplicationInsights;
    };
    /**
     * Enables verbose debug logging
     * @returns {ApplicationInsights} this class
     */
    ApplicationInsights.enableVerboseLogging = function (enableWarningLogging) {
        if (enableWarningLogging === void 0) { enableWarningLogging = true; }
        Logging.enableDebug = true;
        Logging.disableWarnings = !enableWarningLogging;
        return ApplicationInsights;
    };
    /**
     * Disables verbose debug and warning logging
     */
    ApplicationInsights.disableConsoleLogging = function () {
        Logging.enableDebug = false;
        Logging.disableWarnings = true;
        return ApplicationInsights;
    };
    /**
      * Disposes the default client and all the auto collectors so they can be reinitialized with different configuration
      */
    ApplicationInsights.dispose = function () {
        ApplicationInsights.client = null;
        ApplicationInsights._isStarted = false;
        if (ApplicationInsights._console) {
            ApplicationInsights._console.dispose();
        }
        if (ApplicationInsights._exceptions) {
            ApplicationInsights._exceptions.dispose();
        }
        if (ApplicationInsights._performance) {
            ApplicationInsights._performance.dispose();
        }
        if (ApplicationInsights._serverRequests) {
            ApplicationInsights._serverRequests.dispose();
        }
        if (ApplicationInsights._clientRequests) {
            ApplicationInsights._clientRequests.dispose();
        }
    };
    ApplicationInsights._isConsole = true;
    ApplicationInsights._isExceptions = true;
    ApplicationInsights._isPerformance = true;
    ApplicationInsights._isRequests = true;
    ApplicationInsights._isDependencies = true;
    ApplicationInsights._isOfflineMode = false;
    ApplicationInsights._isCorrelating = false;
    ApplicationInsights._isStarted = false;
    return ApplicationInsights;
}());
module.exports = ApplicationInsights;
